;/*************************************************************************
; Program			:Write/Read a 64 kbit EEPROM 24LC256
; Pin Information	:SDA = PC1, SCL = PC0
; CPU Control		:90S8535
; File name			:e24lc256.asm
; Assembler			:AVR Studio 4.05
; 
;**************************************************************************

.include "8535def.inc"			

			.org   	$000 
    		rjmp  reset				;Reset Handle

;/************************
; Define Register
;/************************
.def		counter1  	= r16
.def		counter2	= r17
.def		temp		= r19
.def		data		= r21
.def		adress		= r23

;/***********************
; Define I/O Port Pin
;/***********************
.equ		SCL		= 0
.equ		SDA		= 1
.equ 		EEP_ADRW	= 0xA8		;address write
.equ		EEP_ADRR	= 0xA9  	;address read
.equ		BUF_IN		= $60
.equ		BUF_OUT		= $70
;/*******************
; Main Program
;/*******************
reset:      ldi	temp,low(RAMEND)
            out	SPL,temp          	;init Stack Pointer     
            ldi	temp,high(RAMEND)
            out	SPH,temp        	
main:		sbi	UCR,TXEN
			sbi	UCR,RXEN
			ldi	temp,51				;baudrate = 9600
			out	UBRR,temp

E24LC256:	ser	temp
			out	DDRC,temp
			out	PORTC,temp
			ldi	ZH,high(EEPTB*2)
			ldi	ZL,low(EEPTB*2)
			rcall	Intro
			clr	counter1

;/***********************************
; get a character from serial port 
; and Write it into EEPROM.
;/***********************************
EEP_WR:		clr	XH
			clr	XL
			rcall	RX_Byte			;get data from keyboard
			rcall	TX_Byte			;display it
			cpi	data,0x0D			;if ENTER key is pressed,
			breq	EEP_RD			;program'll goto EEP_RD label

;/**************************
; write a byte to EEPROM
;/**************************
			rcall	I2C_Start		;start condition
			rcall	I2C_WR			;send address and set R/W bit=0 
			rcall	out				;send data to the EEPROM
			rcall	I2C_stop		;stop condition
			rcall	delay5ms
			adiw	XL,1			;increment address of the EEPROM

;/********************************************
; get characters from serial port 
; and Write it into EEPROM, hit ENTER to exit. 
;/********************************************
EEP_WR1:	rcall	RX_Byte
			rcall	TX_Byte

			rcall	I2C_Start
			rcall	I2C_WR
			rcall	out
			rcall	I2C_Stop
			rcall	delay5ms
			adiw	XL,1			;increment address of the EEPROM
			cpi	data,0x0D
			brne	EEP_WR1
			rcall	New_Line
			rcall	New_Line		

;/***********************************************
; Read data from EEPROM and send to serial port.
;/*********************************************** 		
EEP_RD:		clr	XH
			clr	XL
EEP_RD1:	rcall	I2C_Start		;start condition
			rcall	I2C_WR			;write address and set R/W bit = 0
			rcall	I2C_start		;start again
			ldi	data,EEP_ADRR		;write address and set R/W bit = 1
			rcall	out
EEP_RD2:	rcall	in				;receive 8 bit data from the EEPROM
			rcall	I2C_Stop		;stop condition
			rcall	TX_Byte			;display data 
			adiw	XL,1
			cpi	data,0x0D			;if data isn't 0x0D,receive a new data
			brne	EEP_RD1
			rcall	New_Line
			rjmp	EEP_WR	
					 
;/*****************************
; New line and Cariage return
;/*****************************
New_Line:	ldi	data,0x0A			;send line feed
			rcall	TX_Byte
			ldi	data,0x0D			;send cariage return
			rcall	TX_Byte
			ret

;/********************************
; send address to read or write  
;/********************************
I2C_WR:		push	data
			ldi	data,EEP_ADRW		;MSB 7 bit address  + WR bit
			rcall	out
			mov	data,XH				;high byte address  
			rcall	out
			mov	data,XL				;low byte address  
			rcall	out
			pop	data
			ret

;/***************************
; I2C start and stop pulse 
;/***************************
I2C_Start:	sbi	PORTC,SDA
			sbi	PORTC,SCL
			rcall	I2C_delay
			cbi	PORTC,SDA
			rcall	I2C_delay
			cbi	PORTC,SCL
			ret

I2C_Stop:	cbi	PORTC,SDA
			rcall	I2C_delay
			sbi	PORTC,SCL
			rcall	I2C_delay
			sbi	PORTC,SDA
			ret

;/*******************************
; write 8 bit data to SDA line
;/*******************************
out:		push	data
			ldi	counter1,8			;8 bit  data
out1:		sbi	PORTC,SDA		 
			rol	data
			brcs	out2			;if c flag=1 set SDA line else clear SDA line
			cbi	PORTC,SDA
out2:		nop
			sbi	PORTC,SCL
			rcall	I2C_delay
			cbi	PORTC,SCL
			rcall	I2C_delay
			dec	counter1
			brne	out1
			sbi	PORTC,SDA
			nop
			sbi	PORTC,SCL
			rcall	I2C_delay
			sbic	PINC,SDA		;if SDA line=1,begin write old data again
			rjmp	out3
			cbi	PORTC,SCL
			pop	data
			ret
out3:		pop	data
			rjmp	out

;/********************************
; receive 1 byte from SDA line
;/******************************** 
in:			push	counter1
			ldi	counter1,8			;8 bit data
			sbi	PORTC,SDA			;set SDA line for receiving new data
in1:		nop
			sbi	PORTC,SCL
			sec
			sbis	PINC,SDA		;set c flag when SDA line = 1
			clc
			rol	data				;rotate left
			cbi	PORTC,SCL
			rcall	I2C_delay
			dec	counter1
			brne	in1			
			sbi	PORTC,SDA			;send Acknowledge bit
			nop
			sbi	PORTC,SCL
			rcall	I2C_delay
			cbi	PORTC,SCL
			pop	counter1
			ret

;/************************
; Send and Receicve data
;/************************
Intro:		lpm
			tst	r0
			breq	end_sub
TX232:		sbis	USR,UDRE
			rjmp	TX232
			out	UDR,r0
			adiw	ZL,1
			rjmp	Intro
end_sub:	ret		

TX_Byte:	sbis	USR,UDRE
			rjmp	TX_Byte
			out	UDR,data
			ret

RX_Byte:	sbis	USR,RXC
			rjmp	RX_Byte
			in	data,UDR
			ret
;/*****************
; delay time
;/*****************
I2C_delay:	push	counter1
			ldi    	counter1,5
I2C_delay_1: 	
			dec    	counter1
            brne   	I2C_delay_1
			pop	counter1
			ret          

delay1ms:	push	counter1
			push	counter2
			ldi	counter1,8
delay1ms_1: ldi    	counter2,250
delay1ms_2: nop
			dec    	counter2
            brne   	delay1ms_2
            dec    	counter1
            brne   	delay1ms_1
            pop	counter2
			pop	counter1
			ret          

delay5ms:	push	counter1
			push	counter2
			ldi    	counter1,40
delay5ms_1: ldi    	counter2,250
delay5ms_2: nop
			dec    	counter2
            brne   	delay5ms_2
            dec    	counter1
            brne   	delay5ms_1
            pop	counter2
			pop	counter1
			ret          

EEPTB:		.db	0x0a,0x0d,"Test 256 kBit EEPROM 24LC256",0x0a,0x0d
			.db	"Press any key to write data or 'ENTER' key to read old data. ",0x0a,0x0d,0
           	 